local json = require("json")
local sha2 = require("sha2")

function GetVersion()
  return 2
end

function GetCommands()
  return json.encode(
    {
      {
        type = "api",
        data = json.encode({ command = "getdeviceconfig", ipc = true }),
        func = "HandleGetDeviceConfig"
      },
      {
        type = "api",
        data = json.encode({ command = "GetUiConfiguration" }),
        func = "HandleGetUiConfig"
      },
      {
        type = "api",
        data = json.encode({ command = "GetGiraOneDevices" }),
        func = "HandleGetGiraOneDevices"
      }
    }
  )
end

function HandleGetDeviceConfig(response)
  local j = json.decode(response)
  local devcfg = GetFromTable(j, "response", "deviceConfig", "ipc")
  if devcfg then
    local fwv = GetValue(devcfg, "CurrentFirmwareVersion")
    local manu = GetValue(devcfg, "ManufacturerId")
    local device = GetValue(devcfg, "DeviceId")
    local revision = GetValue(devcfg, "ManufacturerRevision", "unknown")
    local project = GetValue(devcfg, "Project")

    return json.encode({
      firmwareVersion = fwv,
      model = (manu .. device),
      manufacturerRevision = revision,
      projectHash = sha2.hash256(project),
    })
  end

  return ""
end

function HandleGetUiConfig(response)
  local j = json.decode(response)
  local cfg = GetFromTable(j, "response", "config")
  if cfg then
    local functions = {}
    for _, chv in ipairs(cfg) do
      local functionType = GetFromTable(chv, "functionType")
      local channelType = GetFromTable(chv, "channelType")
      local channelTypeId = GetFromTable(chv, "channelTypeId")
      if (functionType and channelType and channelTypeId) then
        local key = functionType .. ":" .. channelType .. ":" .. channelTypeId
        local func = functions[key]
        if not func then
          func = { functionType = functionType, channelType = channelType, channelTypeId = channelTypeId, count = 0 }
        end
        local count = func['count']
        count = count + 1
        func['count'] = count
        functions[key] = func
      end
    end

    local jsonFunctions = {}

    for _, func in pairs(functions) do
      table.insert(jsonFunctions, func)
    end

    return json.encode({ functions = jsonFunctions })
  end

  return ""
end

function HandleGetGiraOneDevices(response)
  local j = json.decode(response)
  local devResponse = GetFromTable(j, "response", "devices")
  if devResponse then
    -- empty json object {} can't be encoded with json.lua, so handle this special case here
    if type(devResponse) ~= "table" or next(devResponse) == nil then
      return '{"devices":{}}'
    end

    return json.encode({ devices = devResponse })
  end

  return ""
end

function GetFromTable(tbl, key, ...)
  return not key and tbl or tbl[key] and GetFromTable(tbl[key], ...)
end

function GetValue(tbl, key, default)
  local value = tbl[key]
  if value == nil or value == "" then
    return default == nil and "" or default
  end
  return value
end
