local json = require("json")
local sha2 = require("sha2")

function GetVersion()
  return 4
end

function GetCommands()
  return json.encode(
    {
      {
        type = "api",
        data = json.encode({ command = "getdeviceconfig", ipc = true }),
        func = "HandleGetDeviceConfig"
      },
      {
        type = "api",
        data = json.encode({ command = "GetUiConfiguration" }),
        func = "HandleGetUiConfig"
      }
    }
  )
end

function HandleGetDeviceConfig(response)
  local j = json.decode(response)
  local devcfg = GetFromTable(j, "response", "deviceConfig", "ipc")
  if devcfg then
    local fwv = GetValue(devcfg, "CurrentFirmwareVersion")
    local manu = GetValue(devcfg, "ManufacturerId")
    local device = GetValue(devcfg, "DeviceId")
    local revision = GetValue(devcfg, "ManufacturerRevision", "unknown")
    local project = GetValue(devcfg, "Project")

    return json.encode({
      firmwareVersion = fwv,
      model = (manu .. device),
      manufacturerRevision = revision,
      projectHash = sha2.hash256(project),
    })
  end

  return ""
end

function HandleGetUiConfig(response)
  local j = json.decode(response)
  local cfg = GetFromTable(j, "response", "config")
  if cfg then
    local functions = {}
    for _, chv in ipairs(cfg) do
      local functionType = GetFromTable(chv, "functionType")
      local channelType = GetFromTable(chv, "channelType")
      if (functionType and channelType) then
        local key = functionType .. ":" .. channelType
        local count = functions[key]
        count = count and count + 1 or 1
        functions[key] = count
      end
    end

    local jsonFunctions = {}

    for key, count in pairs(functions) do
      local pos = key:find(':')
      local functionType = key:sub(0, pos - 1)
      local channelType = key:sub(pos + 1)

      table.insert(jsonFunctions, { functionType = functionType, channelType = channelType, count = count })
    end

    return json.encode({ functions = jsonFunctions })
  end

  return ""
end

function GetFromTable(tbl, key, ...)
  return not key and tbl or tbl[key] and GetFromTable(tbl[key], ...)
end

function GetValue(tbl, key, default)
  local value = tbl[key]
  if value == nil or value == "" then
    return default == nil and "" or default
  end
  return value
end
